/*
 * $QNXLicenseC:
 * Copyright 2010, QNX Software Systems.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"). You
 * may not reproduce, modify or distribute this software except in
 * compliance with the License. You may obtain a copy of the License
 * at: http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTIES OF ANY KIND, either express or implied.
 *
 * This file may contain contributions from others, either as
 * contributors under the License or as licensors under other terms.
 * Please review this entire file for other proprietary rights or license
 * notices, as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

/*
 * Synaptics TouchCom touchscreen driver
 *
 * Copyright (C) 2017-2020 Synaptics Incorporated. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * INFORMATION CONTAINED IN THIS DOCUMENT IS PROVIDED "AS-IS," AND SYNAPTICS
 * EXPRESSLY DISCLAIMS ALL EXPRESS AND IMPLIED WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
 * AND ANY WARRANTIES OF NON-INFRINGEMENT OF ANY INTELLECTUAL PROPERTY RIGHTS.
 * IN NO EVENT SHALL SYNAPTICS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, PUNITIVE, OR CONSEQUENTIAL DAMAGES ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OF THE INFORMATION CONTAINED IN THIS DOCUMENT, HOWEVER CAUSED
 * AND BASED ON ANY THEORY OF LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, AND EVEN IF SYNAPTICS WAS ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE. IF A TRIBUNAL OF COMPETENT JURISDICTION DOES
 * NOT PERMIT THE DISCLAIMER OF DIRECT DAMAGES OR ANY OTHER DAMAGES, SYNAPTICS'
 * TOTAL CUMULATIVE LIABILITY TO ANY PARTY SHALL NOT EXCEED ONE HUNDRED U.S.
 * DOLLARS.
 */

/**
 * @file syna_tcm2_runtime.h
 *
 * This file abstracts platform-specific headers and C runtime APIs being used
 * on the target platform.
 */

#ifndef _SYNAPTICS_TCM2_C_RUNTIME_H_
#define _SYNAPTICS_TCM2_C_RUNTIME_H_

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdint.h>
#ifndef UNITTEST
#include <pthread.h>
#include <atomic.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/dcmd_input.h>
#include <sys/neutrino.h>
#include <sys/time.h>
#include <hw/i2c.h>
#include <hw/spi-master.h>
#include <hw/inout.h>
#include <input/mtouch_log.h>
#include <input/event_types.h>
#include <input/mtouch_driver.h>
#include <input/parseopts.h>
#else
#include "pthread_header.h"
#include "ipc_header.h"
#include "touchscreen_header.h"
#include "interrupt_header.h"
#endif
#include <fcntl.h>
#include <malloc.h>


/**
 * @section: Log helpers
 *
 * @brief: LOGD
 *         Output the debug message
 *
 * @brief: LOGI
 *         Output the info message
 *
 * @brief: LOGN
 *         Output the notice message
 *
 * @brief: LOGW
 *         Output the warning message
 *
 * @brief: LOGE
 *         Output the error message
 */

/* string shown for the mtouch_log */
#define MTOUCH_DEV                "mtouch-syna"

#define LOGD(log, ...) \
    mtouch_debug(MTOUCH_DEV, "[  debug] %s: " log, __func__, ##__VA_ARGS__);
#define LOGI(log, ...) \
    mtouch_info(MTOUCH_DEV, "[   info] %s: " log, __func__, ##__VA_ARGS__)
#define LOGN(log, ...) \
    mtouch_info(MTOUCH_DEV, "[   info] %s: " log, __func__, ##__VA_ARGS__)
#define LOGW(log, ...) \
    mtouch_info(MTOUCH_DEV, "[warning] %s: " log, __func__, ##__VA_ARGS__)
#define LOGE(log, ...) \
    mtouch_error(MTOUCH_DEV, "[  error] %s: " log, __func__, ##__VA_ARGS__)


/**
 * @section: NULL pointer checking
 *
 * @brief: _CHECK_POINTER
 *         Ensure the given pointer is valid
 */
#define _CHECK_POINTER(_in_ptr) \
    do { \
        if (!_in_ptr) { \
            mtouch_error(MTOUCH_DEV, "[  error] %s: Invalid Pointer", __func__); \
            return -EINVAL; \
        } \
    } while (0)


/**
 * @section: Error Codes returned
 *           Functions usually return 0 or positive value on success.
 *           Thus, please defines negative value here.
 */
#define _EIO        (-EIO)       /* I/O errors */
#define _ENOMEM     (-ENOMEM)    /* Out of memory */
#define _EINVAL     (-EINVAL)    /* Invalid parameters */
#define _ENODEV     (-ENODEV)    /* No such device */
#define _ETIMEDOUT  (-ETIMEDOUT) /* execution timeout */


/**
 * @section: Data Comparison helpers
 *
 * @brief: MAX
 *         Find the maximum value between
 *
 * @brief: MIN:
 *         Find the minimum value between
 *
 * @brief: GET_BIT
 *         Return the value of target bit
 */
#ifndef MAX
#define MAX(a, b) \
    (a > b ? a : b)
#endif

#ifndef MIN
#define MIN(a, b) \
    (a < b ? a : b)
#endif

#define GET_BIT(var, pos) \
    (((var) & (1 << (pos))) >> (pos))


/**
 * @section: C Atomic operations
 *
 * @brief: ATOMIC_SET
 *         Set an atomic data
 *
 * @brief: ATOMIC_GET:
 *         Get an atomic data
 */
typedef unsigned int syna_pal_atomic_t;

#define ATOMIC_SET(atomic, value) \
    (atomic = value)

#define ATOMIC_GET(atomic) \
    (atomic)


/**
 * @section: C Integer Calculation helpers
 *
 * @brief: syna_pal_le2_to_uint
 *         Convert 2-byte data to an unsigned integer
 *
 * @brief: syna_pal_le4_to_uint
 *         Convert 4-byte data to an unsigned integer
 *
 * @brief: syna_pal_ceil_div
 *         Calculate the ceiling of the integer division
 */

/**
 * syna_pal_le2_to_uint()
 *
 * Convert 2-byte data in little-endianness to an unsigned integer
 *
 * @param
 *    [ in] src: 2-byte data in little-endianness
 *
 * @return
 *    an unsigned integer converted
 */
static inline unsigned int syna_pal_le2_to_uint(const unsigned char *src)
{
    return (unsigned int)src[0] +
        (unsigned int)src[1] * 0x100;
}
/**
 * syna_pal_le4_to_uint()
 *
 * Convert 4-byte data in little-endianness to an unsigned integer
 *
 * @param
 *    [ in] src: 4-byte data in little-endianness
 *
 * @return
 *    an unsigned integer converted
 */
static inline unsigned int syna_pal_le4_to_uint(const unsigned char *src)
{
    return (unsigned int)src[0] +
        (unsigned int)src[1] * 0x100 +
        (unsigned int)src[2] * 0x10000 +
        (unsigned int)src[3] * 0x1000000;
}
/**
 * syna_pal_ceil_div()
 *
 * Calculate the ceiling of the integer division
 *
 * @param
 *    [ in] dividend: the dividend value
 *    [ in] divisor:  the divisor value
 *
 * @return
 *    the ceiling of the integer division
 */
static inline unsigned int syna_pal_ceil_div(unsigned int dividend,
        unsigned int divisor)
{
    return (dividend + divisor - 1) / divisor;
}


/**
 * @section: C Runtime for Memory Management helpers
 *
 * @brief: syna_pal_mem_calloc
 *         Allocate a block of memory space
 *
 * @brief: syna_pal_mem_free
 *         Deallocate a block of memory previously allocated
 *
 * @brief: syna_pal_mem_set
 *         Fill memory with a constant byte
 *
 * @brief: syna_pal_mem_cpy
 *         Ensure the safe size before doing memory copy
 */

/**
 * syna_pal_mem_calloc()
 *
 * Allocates a block of memory for an array of 'num' elements,
 * each of them has 'size' bytes long, and initializes all its bits to zero.
 *
* @param
 *    [ in] num:  number of elements for an array
 *    [ in] size: number of bytes for each elements
 *
 * @return
 *    On success, a pointer to the memory block allocated by the function.
 */
static inline void *syna_pal_mem_alloc(unsigned int num, unsigned int size)
{
    if ((int)(num * size) <= 0) {
        LOGE("Invalid parameter\n");
        return NULL;
    }

    return calloc(num, size);
}
/**
 * syna_pal_mem_free()
 *
 * Deallocate a block of memory previously allocated.
 *
 * @param
 *    [ in] ptr: a memory block  previously allocated
 *
 * @return
 *    none.
 */
static inline void syna_pal_mem_free(void *ptr)
{
    if (ptr)
        free(ptr);
}
/**
 * syna_pal_mem_set()
 *
 * Fill memory with a constant byte
 *
 * @param
 *    [ in] ptr: pointer to a memory block
 *    [ in] c:   the constant value
 *    [ in] n:   number of byte being set
 *
 * @return
 *    none.
 */
static inline void syna_pal_mem_set(void *ptr, int c, unsigned int n)
{
    memset(ptr, c, n);
}
/**
 * syna_pal_mem_cpy()
 *
 * Ensure the safe size before copying the values of num bytes from the
 * location to the memory block pointed to by destination.
 *
 * @param
 *    [out] dest:      pointer to the destination space
 *    [ in] dest_size: size of destination array
 *    [ in] src:       pointer to the source of data to be copied
 *    [ in] src_size:  size of source array
 *    [ in] num:       number of bytes to copy
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static inline int syna_pal_mem_cpy(void *dest, unsigned int dest_size,
        const void *src, unsigned int src_size, unsigned int num)
{
    
	if (src == NULL) {
		LOGD("src null");
		return -1;
	}
	
	if (dest == NULL) {
		LOGD("dest null");
        return -1;
	}
	

    if (num > dest_size || num > src_size) {
        LOGE("Invalid size. src:%d, dest:%d, num:%d\n",
            src_size, dest_size, num);
        return -1;
    }

    memcpy((void *)dest, (const void *)src, num);

    return 0;
}


/**
 * @section: C Runtime for Muxtex Control helpers
 *
 * @brief: syna_pal_mutex_alloc
 *         Create a mutex object
 *
 * @brief: syna_pal_mutex_free
 *         Release the mutex object previously allocated
 *
 * @brief: syna_pal_mutex_lock
 *         Lock the mutex
 *
 * @brief: syna_pal_mutex_unlock
 *         Unlock the mutex previously locked
 */

typedef pthread_mutex_t syna_pal_mutex_t;

/**
 * syna_pal_mutex_alloc()
 *
 * Create a mutex object.
 *
 * @param
 *    [out] ptr: pointer to the mutex handle being allocated
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static inline int syna_pal_mutex_alloc(syna_pal_mutex_t *ptr)
{
    pthread_mutex_init((pthread_mutex_t *)ptr, NULL);
    return 0;
}
/**
 * syna_pal_mutex_free()
 *
 * Release the mutex object previously allocated.
 *
 * @param
 *    [ in] ptr: mutex handle previously allocated
 *
 * @return
 *    none.
 */
static inline void syna_pal_mutex_free(syna_pal_mutex_t *ptr)
{
    pthread_mutex_destroy((pthread_mutex_t *)ptr);
}
/**
 * syna_pal_mutex_lock()
 *
 * Acquire/lock the mutex.
 *
 * @param
 *    [ in] ptr: a mutex handle
 *
 * @return
 *    none.
 */
static inline void syna_pal_mutex_lock(syna_pal_mutex_t *ptr)
{
    pthread_mutex_lock((pthread_mutex_t *)ptr);
}
/**
 * syna_pal_mutex_unlock()
 *
 * Unlock the locked mutex.
 *
 * @param
 *    [ in] ptr: a mutex handle
 *
 * @return
 *    none.
 */
static inline void syna_pal_mutex_unlock(syna_pal_mutex_t *ptr)
{
    pthread_mutex_unlock((pthread_mutex_t *)ptr);
}


/**
 * @section: C Runtime for Completion Event
 *
 * @brief: syna_pal_completion_alloc
 *         Allocate a completion event
 *
 * @brief: syna_pal_completion_free
 *         Release the completion event previously allocated
 *
 * @brief: syna_pal_completion_complete
 *         Complete the completion event being waiting for
 *
 * @brief: syna_pal_completion_reset
 *         Reset or reinitialize the completion event
 *
 * @brief: syna_pal_completion_wait_for
 *         Wait for the completion event
 */

struct completion_s {
    pthread_cond_t cond;
    pthread_mutex_t mutex;
    pthread_condattr_t condattr;
    bool is_signal;
};

typedef struct completion_s syna_pal_completion_t;

/**
 * syna_pal_completion_alloc()
 *
 * Allocate a completion event, and the default state is not set.
 * Caller must reset the event before each use.
 *
 * @param
 *    [out] ptr: pointer to the completion handle being allocated
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static inline int syna_pal_completion_alloc(syna_pal_completion_t *ptr)
{
    struct completion_s *completion = (struct completion_s *)ptr;

    pthread_condattr_init(&completion->condattr);
    pthread_condattr_setclock(&completion->condattr, CLOCK_MONOTONIC);

    pthread_mutex_init(&completion->mutex, NULL);
    pthread_cond_init(&completion->cond, &completion->condattr);

    pthread_mutex_lock(&completion->mutex);
    completion->is_signal = false;
    
    pthread_mutex_unlock(&completion->mutex);

    return 0;
}
/**
 * syna_pal_completion_free()
 *
 * Release the completion event previously allocated
 *
 * @param
 *    [ in] ptr: the completion event previously allocated
 event
 * @return
 *    none.
 */
static inline void syna_pal_completion_free(syna_pal_completion_t *ptr)
{
    struct completion_s *completion = (struct completion_s *)ptr;

    pthread_condattr_destroy(&completion->condattr);
    pthread_cond_destroy(&completion->cond);
    pthread_mutex_destroy(&completion->mutex);
}
/**
 * syna_pal_completion_complete()
 *
 * Complete the completion event being waiting for
 *
 * @param
 *    [ in] ptr: the completion event
 *
 * @return
 *    none.
 */
static inline void syna_pal_completion_complete(syna_pal_completion_t *ptr)
{
    struct completion_s *completion = (struct completion_s *)ptr;

    pthread_mutex_lock(&completion->mutex);

    pthread_cond_signal(&completion->cond);
    completion->is_signal = true;

    pthread_mutex_unlock(&completion->mutex);
}
/**
 * syna_pal_completion_reset()
 *
 * Reset or reinitialize the completion event
 *
 * @param
 *    [ in] ptr: the completion event
 *
 * @return
 *    none.
 */
static inline void syna_pal_completion_reset(syna_pal_completion_t *ptr)
{
    struct completion_s *completion = (struct completion_s *)ptr;

    pthread_mutex_lock(&completion->mutex);

    completion->is_signal = false;

    pthread_mutex_unlock(&completion->mutex);

    return;
}
/**
 * syna_pal_completion_wait_for()
 *
 * Wait for the completion event during the given time slot
 *
 * @param
 *    [ in] ptr:        the completion event
 *    [ in] timeout_ms: time frame in milliseconds
 *
 * @return
 *    0 if a signal is received; otherwise, on timeout or error occurs.
 */
static inline int syna_pal_completion_wait_for(syna_pal_completion_t *ptr,
        unsigned int timeout_ms)
{
    int retval = 0;
    struct timespec t_ts;
    struct completion_s* completion = (struct completion_s*)ptr;

    /* get the real time */
    clock_gettime(CLOCK_MONOTONIC, &t_ts);

    /* add the timeout */
    t_ts.tv_sec += (timeout_ms / 1000);
    t_ts.tv_nsec += (timeout_ms % 1000) * 1000000L;

    if (t_ts.tv_nsec >= 1000000000L) {
        t_ts.tv_sec += 1;
        t_ts.tv_nsec -= 1000000000L;
    }

    pthread_mutex_lock(&completion->mutex);

   /* Use a loop to recheck the condition */
    while (!completion->is_signal) {
        retval = pthread_cond_timedwait(&completion->cond,
            &completion->mutex,
            &t_ts);

        if (retval != 0) {
            break;
        }
    }
    if ((retval == 0) && (completion->is_signal)) {
        pthread_mutex_unlock(&completion->mutex);
           return 0;
    }
    pthread_mutex_unlock(&completion->mutex);
    if (retval == ETIMEDOUT) {
        return _ETIMEDOUT;
    }
    else {
          LOGE("Fail to wait on the completion event, error:%d\n", retval);
              return _EIO;
    }
}


/**
 * @section: C Runtime to Pause the Execution
 *
 * @brief: syna_pal_sleep_ms
 *         Sleep for a fixed amount of time in milliseconds
 *
 * @brief: syna_pal_sleep_us
 *         Sleep for a range of time in microseconds
 *
 * @brief: syna_pal_busy_delay_ms
 *         Busy wait for a fixed amount of time in milliseconds
 */

/**
 * syna_pal_sleep_ms()
 *
 * Sleep for a fixed amount of time in milliseconds
 *
 * @param
 *    [ in] time_ms: time frame in milliseconds
 *
 * @return
 *    none.
 */
static inline void syna_pal_sleep_ms(int time_ms)
{
    if (usleep(time_ms * 1000)) {
		LOGE("Fail");
	}
}
/**
 * syna_pal_sleep_us()
 *
 * Sleep for a range of time in microseconds
 *
 * @param
 *    [ in] time_us_min: the min. time frame in microseconds
 *    [ in] time_us_max: the max. time frame in microseconds
 *
 * @return
 *    none.
 */
static inline void syna_pal_sleep_us(int time_us_min, int time_us_max)
{
    if (usleep((time_us_min + time_us_max) / 2)) {
		LOGE("Fail");
	}
}
/**
 * syna_pal_busy_delay_ms()
 *
 * Busy wait for a fixed amount of time in milliseconds
 *
 * @param
 *    [ in] time_ms: time frame in milliseconds
 *
 * @return
 *    none.
 */
static inline void syna_pal_busy_delay_ms(int time_ms)
{
    if (usleep(time_ms * 1000) != 0) {
		LOGE("Fail");
	}
}


/**
 * @section: C Runtime for String operations
 *
 * @brief: syna_pal_str_len
 *         Return the length of C string
 *
 * @brief: syna_pal_str_cpy:
 *         Ensure the safe size before doing C string copy
 *
 * @brief: syna_pal_str_cmp:
 *         Compare whether the given C strings are equal or not
 */

/**
 * syna_pal_str_len()
 *
 * Return the length of C string
 *
 * @param
 *    [ in] str:  an array of characters
 *
 * @return
 *    the length of given string
 */
static inline unsigned int syna_pal_str_len(const char *str)
{
    return (unsigned int)strlen(str);
}
/**
 * syna_pal_str_cpy()
 *
 * Copy the C string pointed by source into the array pointed by destination.
 *
 * @param
 *    [ in] dest:      pointer to the destination C string
 *    [ in] dest_size: size of destination C string
 *    [out] src:       pointer to the source of C string to be copied
 *    [ in] src_size:  size of source C string
 *    [ in] num:       number of bytes to copy
 *
 * @return
 *    0 on success; otherwise, on error.
 */
static inline int syna_pal_str_cpy(char *dest, unsigned int dest_size,
        const char *src, unsigned int src_size, unsigned int num)
{
    if (dest == NULL || src == NULL)
        return -1;

    if (num > dest_size || num > src_size) {
        LOGE("Invalid size. src:%d, dest:%d, num:%d\n",
            src_size, dest_size, num);
        return -1;
    }

    strncpy(dest, src, num);

    return 0;
}
/**
 * syna_pal_str_cmp()
 *
 * Compares up to num characters of the C string str1 to those of the
 * C string str2.
 *
 * @param
 *    [ in] str1: C string to be compared
 *    [ in] str2: C string to be compared
 *    [ in] num:  number of characters to compare
 *
 * @return
 *    0 if both strings are equal; otherwise, not equal.
 */
static inline int syna_pal_str_cmp(const char *str1, const char *str2,
        unsigned int num)
{
    return strncmp(str1, str2, num);
}
/**
 * syna_pal_hex_to_uint()
 *
 * Convert the given string in hex to an integer returned
 *
 * @param
 *    [ in] str:    C string to be converted
 *    [ in] length: target length
 *
 * @return
 *    An integer converted
 */
static inline unsigned int syna_pal_hex_to_uint(char *str, int length)
{
    unsigned int result = 0;
    char *ptr = NULL;

    for (ptr = str; ptr != str + length; ++ptr) {
        result <<= 4;
        if (*ptr >= 'A')
            result += *ptr - 'A' + 10;
        else
            result += *ptr - '0';
    }

    return result;
}

/**
 * @section: C Runtime for Checksum Calculation
 *
 * @brief: syna_pal_crc32
 *         Calculates the CRC32 value
 */

/**
 * syna_pal_crc32()
 *
 * Calculates the CRC32 value of the data
 *
 * @param
 *    [ in] seed: the previous crc32 value
 *    [ in] data: byte data for the calculation
 *    [ in] len:  the byte length of the data.
 *
 * @return
 *    0 if both strings are equal; otherwise, not equal.
 */
static inline unsigned int syna_pal_crc32(unsigned int seed,
        const char *data, unsigned int len)
{
    unsigned int crc;
    unsigned char *p_current;

    /* polynomial lookup */
    const unsigned int lookup[16] =
    {
        0x00000000, 0x1DB71064, 0x3B6E20C8, 0x26D930AC,
        0x76DC4190, 0x6B6B51F4, 0x4DB26158, 0x5005713C,
        0xEDB88320, 0xF00F9344, 0xD6D6A3E8, 0xCB61B38C,
        0x9B64C2B0, 0x86D3D2D4, 0xA00AE278, 0xBDBDF21C
    };

    crc = seed;
    p_current = (unsigned char *)&data[0];

    while (len--)
    {
        crc = lookup[(crc ^ *p_current) & 0x0F] ^ (crc >> 4);
        crc = lookup[(crc ^ (*p_current >> 4)) & 0x0F] ^ (crc >> 4);
        p_current++;
    }

    return crc;
}

#endif /* end of _SYNAPTICS_TCM2_C_RUNTIME_H_ */
